#!/usr/bin/env bash
# ═══════════════════════════════════════════════════════════════════════════════
# HLAT — Linux Installer
# ═══════════════════════════════════════════════════════════════════════════════
#
# What this script does:
#
#   1. Detects whether HLAT is already installed (system-wide or user-only).
#      If found, offers to uninstall it and exit.
#
#   2. If not installed (or you skipped uninstall), asks where to install:
#
#      System-wide  (requires sudo / root):
#        AppImage  →  /usr/local/bin/hlat
#        Icon      →  /usr/share/pixmaps/hlat.png
#        Desktop   →  /usr/share/applications/hlat.desktop
#
#      Current user only  (no root required):
#        AppImage  →  ~/.local/bin/hlat
#        Icon      →  ~/.local/share/icons/hlat.png
#        Desktop   →  ~/.local/share/applications/hlat.desktop
#
#   After installation HLAT will appear in your application launcher and
#   can be launched from a terminal with:  hlat
#
#   Note: For a user install, ~/.local/bin must be on your PATH.
#   Most modern distributions (Ubuntu, Fedora, Arch, etc.) add it
#   automatically on next login if the directory exists.
#
# Usage:
#   bash install.sh
#
# ═══════════════════════════════════════════════════════════════════════════════
set -euo pipefail

# ── Paths ─────────────────────────────────────────────────────────────────────

SYSTEM_BIN="/usr/local/bin/hlat"
SYSTEM_ICON="/usr/share/pixmaps/hlat.png"
SYSTEM_DESKTOP="/usr/share/applications/hlat.desktop"

USER_BIN="${HOME}/.local/bin/hlat"
USER_ICON="${HOME}/.local/share/icons/hlat.png"
USER_DESKTOP="${HOME}/.local/share/applications/hlat.desktop"

# ── Locate bundled files ───────────────────────────────────────────────────────

SCRIPT_DIR="$(dirname "$(readlink -f "${BASH_SOURCE[0]}")")"

# Use bash glob instead of find — handles special characters (parentheses,
# spaces, etc.) in directory names without issues.
shopt -s nullglob
_appimages=("${SCRIPT_DIR}"/*.AppImage)
shopt -u nullglob

if [ ${#_appimages[@]} -eq 0 ]; then
    echo "Error: No .AppImage file found next to install.sh."
    echo "Make sure you extracted the full release archive."
    exit 1
fi

APPIMAGE="${_appimages[0]}"
ICON="${SCRIPT_DIR}/hlat.png"

# ── Helpers ───────────────────────────────────────────────────────────────────

confirm() {
    # Usage: confirm "Prompt" && do_something
    local prompt="${1} [y/N] "
    local answer
    read -r -p "${prompt}" answer
    [[ "${answer}" =~ ^[Yy]$ ]]
}

update_desktop_db() {
    local dir="$1"
    if command -v update-desktop-database &>/dev/null; then
        update-desktop-database "${dir}" 2>/dev/null || true
    fi
}

update_icon_cache() {
    local dir="$1"
    if command -v gtk-update-icon-cache &>/dev/null; then
        gtk-update-icon-cache -f -t "${dir}" 2>/dev/null || true
    fi
}

# ── Check for existing installation ───────────────────────────────────────────

SYSTEM_INSTALLED=false
USER_INSTALLED=false

[ -f "${SYSTEM_BIN}" ]  && SYSTEM_INSTALLED=true
[ -f "${USER_BIN}" ]    && USER_INSTALLED=true

if ${SYSTEM_INSTALLED} || ${USER_INSTALLED}; then
    echo ""
    echo "HLAT is already installed:"
    ${SYSTEM_INSTALLED} && echo "  System-wide : ${SYSTEM_BIN}"
    ${USER_INSTALLED}   && echo "  User        : ${USER_BIN}"
    echo ""

    if confirm "Do you want to uninstall it?"; then
        echo ""

        if ${SYSTEM_INSTALLED}; then
            echo "==> Removing system-wide installation (requires sudo)..."
            sudo rm -f "${SYSTEM_BIN}" "${SYSTEM_ICON}" "${SYSTEM_DESKTOP}"
            update_desktop_db "$(dirname "${SYSTEM_DESKTOP}")"
            echo "    Removed."
        fi

        if ${USER_INSTALLED}; then
            echo "==> Removing user installation..."
            rm -f "${USER_BIN}" "${USER_ICON}" "${USER_DESKTOP}"
            update_desktop_db "$(dirname "${USER_DESKTOP}")"
            echo "    Removed."
        fi

        echo ""
        echo "HLAT has been uninstalled."
        exit 0
    fi

    echo ""
    echo "Uninstall skipped. Proceeding with installation over existing files."
    echo ""
fi

# ── Choose install scope ───────────────────────────────────────────────────────

echo ""
echo "Install HLAT for:"
echo "  1) System-wide  (all users, requires sudo)"
echo "  2) Current user only  (no root required)"
echo ""

SCOPE=""
while true; do
    read -r -p "Choice [1/2]: " SCOPE
    case "${SCOPE}" in
        1|2) break ;;
        *) echo "Please enter 1 or 2." ;;
    esac
done

# ── Install ───────────────────────────────────────────────────────────────────

echo ""

if [ "${SCOPE}" = "1" ]; then
    # ── System-wide ───────────────────────────────────────────────────────────
    echo "==> Installing system-wide (sudo required)..."

    sudo install -Dm755 "${APPIMAGE}"    "${SYSTEM_BIN}"
    sudo install -Dm644 "${ICON}"        "${SYSTEM_ICON}"

    # Write desktop file with correct paths
    sudo tee "${SYSTEM_DESKTOP}" > /dev/null << EOF
[Desktop Entry]
Name=HLAT
Comment=Measure input latency, mouse precision, and polling rate
Exec=${SYSTEM_BIN}
Icon=${SYSTEM_ICON}
Type=Application
Categories=Utility;
Terminal=false
EOF

    update_desktop_db "$(dirname "${SYSTEM_DESKTOP}")"

    echo ""
    echo "HLAT installed system-wide."
    echo "  Launch from your application menu, or type: hlat"

else
    # ── User-only ─────────────────────────────────────────────────────────────
    echo "==> Installing for current user..."

    install -Dm755 "${APPIMAGE}"    "${USER_BIN}"
    install -Dm644 "${ICON}"        "${USER_ICON}"

    mkdir -p "$(dirname "${USER_DESKTOP}")"

    tee "${USER_DESKTOP}" > /dev/null << EOF
[Desktop Entry]
Name=HLAT
Comment=Measure input latency, mouse precision, and polling rate
Exec=${USER_BIN}
Icon=${USER_ICON}
Type=Application
Categories=Utility;
Terminal=false
EOF

    update_desktop_db "$(dirname "${USER_DESKTOP}")"
    update_icon_cache "${HOME}/.local/share/icons"

    echo ""
    echo "HLAT installed for user '${USER}'."
    echo "  Launch from your application menu, or type: hlat"

    # Remind about PATH if ~/.local/bin wasn't already on it
    if [[ ":${PATH}:" != *":${HOME}/.local/bin:"* ]]; then
        echo ""
        echo "  Note: ~/.local/bin is not currently on your PATH."
        echo "  Log out and back in (or run 'source ~/.profile') for"
        echo "  the 'hlat' terminal command to become available."
    fi
fi

echo ""
